from src.tuxedo_tomte_light.state_manager import (
    INSTALLED_DIR,
    mark_installed,
    is_already_installed,
    reset_installed_state,
)


def test_mark_installed_creates_directory_and_file(fs):
    pkg = "testpkg"

    assert mark_installed(pkg) is True

    flag_file = INSTALLED_DIR / pkg

    assert INSTALLED_DIR.exists()
    assert flag_file.exists()

    content = flag_file.read_text().strip()
    assert "T" in content

def test_mark_installed_write_error(fs, monkeypatch):
    pkg = "testpkg"

    fs.makedirs(INSTALLED_DIR, exist_ok=True)

    flag_file = INSTALLED_DIR / pkg

    fs.create_file(flag_file)

    def fake_write_text(*args, **kwargs):
        raise OSError("simulated write failure")

    monkeypatch.setattr(type(flag_file), "write_text", fake_write_text)

    result = mark_installed(pkg)
    assert result is False


def test_mark_installed_creates_directory_if_missing(fs):
    pkg = "auto"

    assert not INSTALLED_DIR.exists()

    mark_installed(pkg)

    assert INSTALLED_DIR.exists()
    assert (INSTALLED_DIR / pkg).exists()


def test_is_already_installed_true(fs):
    pkg = "mypkg"

    fs.makedirs(INSTALLED_DIR, exist_ok=True)
    fs.create_file(INSTALLED_DIR / pkg)

    assert is_already_installed(pkg) is True

def test_is_already_installed_false(fs):
    pkg = "missingpkg"

    fs.makedirs(INSTALLED_DIR, exist_ok=True)

    assert is_already_installed(pkg) is False


def test_reset_installed_state_removes_files(fs):
    fs.makedirs(INSTALLED_DIR, exist_ok=True)

    fs.create_file(INSTALLED_DIR / "a")
    fs.create_file(INSTALLED_DIR / "b")
    fs.create_file(INSTALLED_DIR / "c")

    assert reset_installed_state() is True

    assert INSTALLED_DIR.exists()
    assert list(INSTALLED_DIR.iterdir()) == []

def test_reset_installed_state_missing_dir(fs):
    assert not INSTALLED_DIR.exists()

    result = reset_installed_state()
    assert result is False

def test_reset_installed_state_unlink_error(fs, monkeypatch):
    fs.makedirs(INSTALLED_DIR, exist_ok=True)

    file = INSTALLED_DIR / "pkg"
    fs.create_file(file)

    def fake_unlink(self):
        raise OSError("simulated unlink failure")

    monkeypatch.setattr(type(file), "unlink", fake_unlink)

    result = reset_installed_state()
    assert result is False
