from pathlib import Path

from .file_operations import load_yaml_file
from .logging import logger
from .tuxedo_detection import is_tuxedo_device, is_tuxedo_os


def validate_config(cfg: dict, filename: str) -> bool:
    """Ensures the YAML structure matches Tomte's device-rule schema."""

    logger.debug(f"Validating configuration: {filename}")

    if not isinstance(cfg, dict):
        logger.error(f"{filename} top-level must be a dict")
        return False

    for key, items in cfg.items():
        if not items:
            continue

        if not isinstance(items, list):
            logger.error(f"'{key}' must be a list")
            return False

        for entry in items:
            if not isinstance(entry, dict):
                logger.error(f"entry in '{key}' must be a dict")
                return False

            if len(entry) != 1:
                logger.error(f"entry in '{key}' must have exactly one key")
                return False

            k, v = next(iter(entry.items()))

            # --- PCI ---
            if k == "PCI":
                if not isinstance(v, str) and not (
                    isinstance(v, list) and all(isinstance(x, str) for x in v)
                ):
                    logger.error(
                        f"{filename} PCI value must be a string or list of strings"
                    )
                    return False

            # --- USB ---
            elif k == "USB":
                if not isinstance(v, list) or not all(isinstance(x, str) for x in v):
                    logger.error(f"{filename} USB value must be a list of strings")
                    return False

            # --- GPU_COUNT_min ---
            elif k == "GPU_COUNT_min":
                if not isinstance(v, int):
                    logger.error(f"{filename} GPU_COUNT_min must be an integer")
                    return False

            # --- Unknown keys ---
            else:
                logger.error(f"{filename} unknown device type '{k}'")
                return False
    return True


def merge_configs(configs: list[dict]) -> dict:
    """Later configs override earlier ones entirely."""
    result: dict = {}
    for cfg in configs:
        result.update(cfg)
    return result


def get_config() -> dict:
    """
    Loads and merges all configuration layers in correct priority order.
    Later files fully override earlier ones.
    """

    base_path = Path("/usr/lib/tuxedo-tomte-light")

    files = ["base.yaml"]

    if is_tuxedo_device():
        files.append("tuxedo-devices.yaml")
    if is_tuxedo_os():
        files.append("tuxedo-os.yaml")

    logger.info("Loading TUXEDO Tomte Light configuration layers...")

    loaded: list[dict] = []

    for fname in files:
        full = base_path / fname
        logger.debug(f"Processing: {full}")

        cfg = load_yaml_file(full)
        if not validate_config(cfg, fname):
            return {}
        loaded.append(cfg)

    merged = merge_configs(loaded)

    logger.debug("Final merged configuration:")
    logger.debug(merged)

    return merged
