from pathlib import Path

import yaml

from .logging import logger


def read_file(path: str | Path) -> str | None:
    """Safe file reader with logging, returns None if missing."""
    path = Path(path)

    try:
        return path.read_text(encoding="utf-8").strip()
    except FileNotFoundError:
        logger.debug(f"File not found: {path}")
        return None
    except Exception as e:
        logger.warning(f"Error reading {path}: {e}")
        return None


def load_yaml_file(path: str | Path) -> dict:
    """Load a YAML file safely. Returns {} on empty files but throws on errors."""

    path = Path(path)

    logger.debug(f"Loading YAML file: {path}")

    if not path.exists():
        logger.error(f"{str(path)} file does not exist")
        return {}

    if not path.is_file():
        logger.error(f"{str(path)} is not a regular file")
        return {}

    try:
        with path.open("r", encoding="utf-8") as f:
            data = yaml.safe_load(f)
            if not data:
                logger.error(f"{str(path)} File is empty!")
            return data or {}
    except yaml.YAMLError as e:
        logger.error(f"{str(path)} YAML parsing error: {e}")
        return {}
    except Exception as e:
        logger.error(f"{str(path)} unexpected error: {e}")
        return {}
