import os
import shutil
import subprocess
from abc import ABC, abstractmethod

from .logging import logger


class PackageManager(ABC):
    BIN: str  # Name des Binaries, muss in Subklassen angegeben werden

    @staticmethod
    def _run(args: list[str]) -> tuple[int, str, str]:
        """Executes a system command with full logging and returns (rc, stdout, stderr)."""
        logger.debug(f"Executing command: {' '.join(args)}")

        proc = subprocess.run(
            args,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
            env={
                **os.environ,
                "LANG": "C",
                "LC_ALL": "C",
            },
        )

        logger.debug(f"Return code: {proc.returncode}")
        if proc.stdout:
            logger.debug(f"Stdout:\n{proc.stdout.rstrip()}")
        if proc.stderr:
            logger.debug(f"Stderr:\n{proc.stderr.rstrip()}")

        return proc.returncode, proc.stdout, proc.stderr

    def _run_checked(self, args: list[str]) -> tuple[int, str]:
        """Runs a command and raises an exception on failure."""
        rc, out, err = self._run(args)
        if rc != 0:
            logger.error(f"Command failed: {' '.join(args)}")
        return rc, out

    @staticmethod
    def _normalize_packages(packages: list[str] | str) -> list[str]:
        if isinstance(packages, str):
            logger.debug(
                "Received a single string instead of list[str]. Autocorrecting..."
            )
            return packages.split()
        return packages

    # ---- Abstract API ----

    @abstractmethod
    def install(self, packages: list[str] | str) -> bool: ...
    @abstractmethod
    def update(self) -> bool: ...
    @abstractmethod
    def package_exists(self, pkgname: str) -> bool: ...


class AptManager(PackageManager):
    BIN = "apt-get"

    def install(self, packages: list[str] | str) -> bool:
        packages = self._normalize_packages(packages)
        logger.info(f"Installing via APT (with recommends): {packages}")
        self._run_checked(
            [self.BIN, "install", "-y", "--install-recommends", *packages]
        )
        return True

    def update(self) -> bool:
        logger.info("Updating APT package lists.")
        return self._run_checked([self.BIN, "update"])[0] == 0

    def package_exists(self, pkgname: str) -> bool:
        logger.debug(f"Checking if APT package exists: {pkgname}")
        rc, out, _ = self._run(["apt-cache", "policy", pkgname])

        for line in out.splitlines():
            if line.strip().startswith("Candidate:") and "(none)" not in line:
                return True

        return False


class DnfManager(PackageManager):
    BIN = "dnf"

    def install(self, packages: list[str] | str) -> bool:
        packages = self._normalize_packages(packages)
        logger.info(f"Installing via DNF: {packages}")
        self._run_checked([self.BIN, "-y", "install", *packages])
        return True

    def update(self) -> bool:
        logger.info("Checking for DNF updates.")

        rc, out, err = self._run([self.BIN, "check-update"])

        # DNF returns:
        #   0   = no updates available (success)
        #   100 = updates available (still success)
        #   else = error
        if rc in (0, 100):
            return True

        logger.error(f"DNF check-update failed: rc={rc}")
        return False

    def package_exists(self, pkgname: str) -> bool:
        """
        Reliable DNF package existence check.

        1. Check if package is installed (rpm -q)
        2. Check if package is available (dnf repoquery)
        3. Fallback: dnf list available
        """
        logger.debug(f"Checking if DNF package exists: {pkgname}")
        # Step 1: Installed?
        rc, _, _ = self._run(["rpm", "-q", pkgname])
        if rc == 0:
            return True

        # Step 2: Available? (DNF's official way)
        # repoquery returns 0 only if package exists
        rc, out, err = self._run(["dnf", "-q", "repoquery", pkgname])
        if rc == 0 and pkgname in out:
            return True

        # Step 3: Fallback for minimal DNF builds
        rc, out, err = self._run(["dnf", "-q", "list", "available", pkgname])
        if "Available Packages" in out and pkgname in out:
            return True

        return False


class PacmanManager(PackageManager):
    BIN = "pacman"

    def install(self, packages: list[str] | str) -> bool:
        packages = self._normalize_packages(packages)
        logger.info(f"Installing via Pacman: {packages}")
        self._run_checked([self.BIN, "-S", "--needed", "--noconfirm", *packages])
        return True

    def update(self) -> bool:
        logger.info("Refreshing Pacman repositories.")
        return self._run_checked([self.BIN, "-Sy"])[0] == 0

    def package_exists(self, pkgname: str) -> bool:
        logger.debug(f"Checking if pacman package exists: {pkgname}")
        rc, out, _ = self._run(["pacman", "-Si", pkgname])
        return "Repository" in out


class ZypperManager(PackageManager):
    BIN = "zypper"

    def install(self, packages: list[str] | str) -> bool:
        packages = self._normalize_packages(packages)
        logger.info(f"Installing via Zypper (with recommends): {packages}")
        self._run_checked(
            [self.BIN, "--non-interactive", "install", "--recommends", *packages]
        )
        return True

    def update(self) -> bool:
        logger.info("Refreshing Zypper repositories.")
        return self._run_checked([self.BIN, "--non-interactive", "refresh"])[0] == 0

    def package_exists(self, pkgname: str) -> bool:
        logger.debug(f"Checking if Zyper package exists: {pkgname}")
        rc, out, _ = self._run(["zypper", "search", "--match-exact", pkgname])
        return pkgname in out


def detect_package_manager() -> PackageManager | None:
    """
    Automatically detects the active Linux package manager by scanning all
    PackageManager subclasses and checking their BIN attribute.
    """

    subclasses = PackageManager.__subclasses__()

    priority = ["apt-get", "dnf", "pacman", "zypper"]

    subclasses.sort(
        key=lambda class_name: priority.index(class_name.BIN)
        if getattr(class_name, "BIN", None) in priority
        else 999
    )

    for cls in subclasses:
        binary = getattr(cls, "BIN", None)
        if binary and shutil.which(binary):
            pm = cls()
            logger.info(f"Detected package manager: {cls.__name__} (BIN={binary})")
            return pm

    logger.error("Kein unterstützter Paketmanager gefunden.")
    return None
