from datetime import datetime, timezone
from pathlib import Path

from .logging import logger

STATE_ROOT = Path("/var/lib/tuxedo-tomte-light/state")
INSTALLED_DIR = STATE_ROOT / "installed"


def _ensure_dir(path: Path) -> None:
    """Ensures a directory exists. Raises on fatal errors."""
    try:
        path.mkdir(parents=True, exist_ok=True)
    except Exception as e:
        logger.error(f"Could not create directory {path}: {e}")


def mark_installed(pkg: str) -> bool:
    """
    Mark a package as installed by creating a timestamped flag file.
    """
    logger.debug(f"Marking package as installed: {pkg}")

    _ensure_dir(INSTALLED_DIR)

    flag_file = INSTALLED_DIR / pkg
    timestamp = datetime.now(timezone.utc).isoformat()

    try:
        flag_file.write_text(timestamp + "\n", encoding="utf-8")
        logger.info(f"Created install flag: {flag_file}")
    except Exception as e:
        logger.error(f"Failed to write install flag {flag_file}: {e}")
        return False
    return True


def is_already_installed(pkg: str) -> bool:
    """
    Returns True if the package flag file exists.
    """

    flag_file = INSTALLED_DIR / pkg
    exists = flag_file.exists()

    logger.debug(f"Check installed: {pkg} -> {exists}")

    return exists


def reset_installed_state() -> bool:
    """
    Removes all installation flags.
    """

    if not INSTALLED_DIR.exists():
        logger.info("No installed-state directory present. Nothing to reset.")
        return False

    logger.info("Resetting installed package state flags...")

    for f in INSTALLED_DIR.iterdir():
        if f.is_file():
            try:
                f.unlink()
                logger.debug(f"Removed: {f}")
            except Exception as e:
                logger.error(f"Could not remove {f}: {e}")
                return False
    return True
