import subprocess
import logging
import sys

logger = logging.getLogger(__name__)


def _run_command(cmd: list[str], timeout: int) -> dict[str, str | bool | int | None]:
    """
    Run a system command with timeout and full error handling.
    If a timeout occurs → FATAL → exit(1).
    """

    logger.debug(f"Executing command: {' '.join(cmd)} with timeout={timeout}s")

    try:
        proc = subprocess.run(
            cmd,
            capture_output=True,
            text=True,
            timeout=timeout,
            check=False
        )

        stdout = proc.stdout.strip()
        stderr = proc.stderr.strip()

        return {
            "success": True,
            "returncode": proc.returncode,
            "stdout": stdout,
            "stderr": stderr,
        }

    except subprocess.TimeoutExpired:
        logger.critical(f"Command timed out after {timeout}s: {' '.join(cmd)}")
        sys.exit(1)

    except FileNotFoundError:
        logger.error(f"Command not found: {cmd[0]}")
        return {
            "success": False,
            "returncode": None,
            "stdout": "",
            "stderr": f"{cmd[0]} not found",
        }

    except Exception as e:
        logger.exception(f"Unexpected error while running: {' '.join(cmd)}")
        return {
            "success": False,
            "returncode": None,
            "stdout": "",
            "stderr": str(e),
        }


def update_hardware_information(timeout_seconds: int = 600) -> dict:
    """
    Runs lspci and lsusb.
    If either command hits the timeout → program exits(1).
    """

    logger.info("Updating hardware information...")

    results = {
        "lspci": _run_command(["lspci", "-n"], timeout_seconds),
        "lsusb": _run_command(["lsusb"], timeout_seconds),
    }

    logger.info("Hardware information update completed.")
    return results
