import pytest
import os
import subprocess

from src.tuxedo_tomte_light.package_manager import AptManager

if os.environ.get("PM") not in ['apt', 'apt-get']:
    pytest.skip("Skipping apt tests because PM=apt or PM=apt-get not set", allow_module_level=True)


def run(cmd):
    return subprocess.run(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True)


def test_apt_update_changes_lists():
    pm = AptManager()

    # get timestamp BEFORE update
    lists_dir = "/var/lib/apt/lists"
    before = max(os.path.getmtime(os.path.join(lists_dir, f)) for f in os.listdir(lists_dir))

    pm.update()

    # get timestamp AFTER update
    after = max(os.path.getmtime(os.path.join(lists_dir, f)) for f in os.listdir(lists_dir))

    assert after > before, "apt-get update did not change timestamps in /var/lib/apt/lists"


@pytest.mark.parametrize("pkg", ['nano', 'htop', 'curl'])
def test_apt_install_reinstalls_removed_package(pkg):
    pm = AptManager()

    # ensure installed by removing first
    run(["apt-get", "purge", "-y", pkg])

    assert pm.package_exists(pkg) is True

    rc = run(["dpkg", "-s", pkg]).returncode
    assert rc != 0, "Package should be removed before test"

    # now install via our manager
    pm.install(pkg)

    rc = run(["dpkg", "-s", pkg]).returncode
    assert rc == 0, "Package should be installed after pm.install()"

    assert pm.package_exists(pkg) is True
