import os

import pytest

from src.tuxedo_tomte_light.package_manager import (
    detect_package_manager,
    PackageManager,
)


@pytest.fixture(scope="session")
def pm() -> PackageManager:
    """Detect the system package manager and fail if none is found."""
    manager = detect_package_manager()
    if manager is None:
        pytest.fail("No supported package manager detected in this environment.")
    return manager


def test_detect_package_manager(pm):
    """Ensure detect_package_manager() returns the correct class."""
    assert isinstance(pm, PackageManager), \
        f"detect_package_manager() returned wrong type: {type(pm)}"

@pytest.mark.skipif(os.environ.get("PM") not in ['apt', 'apt-get', 'dnf', 'zypper', 'pacman'],
                    reason="No valid PM= found")
def test_detect_correct_package_manager(pm):
    """Ensure detect_package_manager() returns the correct Package-Manager."""
    assert isinstance(pm, PackageManager), \
        f"detect_package_manager() returned wrong type: {type(pm)}"

    assert pm.BIN.split('-')[0] == os.environ.get("PM").split('-')[0], \
        f"detect_package_manager() returned wrong Package-Manager {pm.BIN}"


def test_update_does_not_fail(pm):
    """Ensure update/refresh actually runs and returns True."""
    result = pm.update()
    assert result is True, "pm.update() should return True"


def test_package_exists_positive(pm):
    """
    Check a well-known package across all distros.
    bash *always* exists.
    """
    exists = pm.package_exists("bash")
    assert exists is True, "bash should exist on all Linux distributions"


def test_package_exists_negative(pm):
    """
    Use a package name that is guaranteed NOT to exist.
    Make sure the package manager returns proper False.
    """
    exists = pm.package_exists("thispackagedoesnotexist-xyz-123-foo-bar")
    assert exists is False, "Nonexistent package should not be detected"

@pytest.mark.parametrize("pkg", ['nano', 'htop', 'curl', 'bash'])
def test_install_changes_state(pm, pkg):
    assert pm.package_exists(pkg) is True, "pm.install() must return True on success"
