import pytest
from src.tuxedo_tomte_light.package_mapper import match_ids


PATTERNS = [
    r"147e:100[0123]",          # 1000–1003
    r"147e:20(15|16|20)",       # 2015, 2016, 2020
    r"147e:300[01]",            # 3000–3001
    r"147e:500[23]",            # 5002–5003
    r"0483:201[56]",            # 2015–2016
    r"1c7a:0603",               # exact match
]


@pytest.mark.parametrize("pattern,dev_id", [
    (r"147e:100[0123]", "147e:1000"),
    (r"147e:100[0123]", "147e:1001"),
    (r"147e:100[0123]", "147e:1002"),
    (r"147e:100[0123]", "147e:1003"),
])
def test_match_147e_100x(pattern, dev_id):
    assert match_ids([dev_id], [pattern]) == [dev_id]


@pytest.mark.parametrize("dev_id", ["147e:2015", "147e:2016", "147e:2020"])
def test_match_147e_20xx(dev_id):
    pattern = r"147e:20(15|16|20)"
    assert match_ids([dev_id], [pattern]) == [dev_id]


@pytest.mark.parametrize("dev_id", ["147e:3000", "147e:3001"])
def test_match_147e_300x(dev_id):
    pattern = r"147e:300[01]"
    assert match_ids([dev_id], [pattern]) == [dev_id]


@pytest.mark.parametrize("dev_id", ["147e:5002", "147e:5003"])
def test_match_147e_500x(dev_id):
    pattern = r"147e:500[23]"
    assert match_ids([dev_id], [pattern]) == [dev_id]


@pytest.mark.parametrize("dev_id", ["0483:2015", "0483:2016"])
def test_match_0483_201x(dev_id):
    pattern = r"0483:201[56]"
    assert match_ids([dev_id], [pattern]) == [dev_id]


def test_match_1c7a_0603():
    assert match_ids(["1c7a:0603"], [r"1c7a:0603"]) == ["1c7a:0603"]



@pytest.mark.parametrize("dev_id", [
    "147e:1004",
    "147e:2014",
    "147e:3002",
    "147e:5001",
    "0483:2014",
    "1c7a:0604",
])
def test_no_false_positives(dev_id):
    assert match_ids([dev_id], PATTERNS) == []



def test_all_patterns_multi_match():
    available_ids = [
        "147e:1000",
        "147e:2020",
        "147e:3001",
        "1c7a:0603",
        "9999:0000",   # kein Treffer
    ]

    result = match_ids(available_ids, PATTERNS)

    assert set(result) == {
        "147e:1000",
        "147e:2020",
        "147e:3001",
        "1c7a:0603",
    }
