import pytest
import pyfakefs # install this dependency or it might overwrite files on your computer

import src
from src.tuxedo_tomte_light.config_parser import validate_config, merge_configs, get_config
from src.tuxedo_tomte_light.package_mapper import match_ids, resolve_mapping, get_os_ids, \
    get_hardware_config_dependend_packages

valid_config_list = [
    {'tuxedo-yt6801': [{'PCI': '1f0a:6801'}], 'tuxedo-fix-ath12k-standby': [{'PCI': '17cb:1107'}],
     'fingerprintreader': [
         {'USB': ['147e:100[0123]', '147e:20(15|16|20)', '147e:300[01]', '147e:500[23]', '0483:201[56]', '1c7a:0603']}],
     'tuxedo-fix-wifi-set-reg-domain': [{'PCI': ['14c3:0608', '14c3:0616']}], 'tuxedo-dgpu-run': [{'GPU_COUNT_min': 2}]}
]

invalid_config_list = [
    {'tuxedo-yt6801': [{'PCI': '1f0a:6801'}], 'tuxedo-fix-ath12k-standby': [{'PCI': '17cb:1107'}],
     'fingerprintreader': [
         {'USB': ['147e:100[0123]', '147e:20(15|16|20)', '147e:300[01]', '147e:500[23]', '0483:201[56]', '1c7a:0603']}],
     'tuxedo-fix-wifi-set-reg-domain': [{'PCI': ['14c3:0608', '14c3:0616']}], 'tuxedo-dgpu-run': [{'GPU_COUNT_min': "2"}]}
]

merge_configs_list = [
    ([
         {'tuxedo-yt6801': [{'PCI': '1f0a:6801'}], 'tuxedo-fix-ath12k-standby': [{'PCI': '17cb:1107'}]},
         {'fingerprintreader': [
         {'USB': ['147e:100[0123]', '147e:20(15|16|20)', '147e:300[01]', '147e:500[23]', '0483:201[56]', '1c7a:0603']}]},
         {'tuxedo-fix-wifi-set-reg-domain': [{'PCI': ['14c3:0608', '14c3:0616']}]},
         {'tuxedo-dgpu-run': [{'GPU_COUNT_min': 2}]}
     ],     {'tuxedo-yt6801': [{'PCI': '1f0a:6801'}], 'tuxedo-fix-ath12k-standby': [{'PCI': '17cb:1107'}],
     'fingerprintreader': [
         {'USB': ['147e:100[0123]', '147e:20(15|16|20)', '147e:300[01]', '147e:500[23]', '0483:201[56]', '1c7a:0603']}],
     'tuxedo-fix-wifi-set-reg-domain': [{'PCI': ['14c3:0608', '14c3:0616']}], 'tuxedo-dgpu-run': [{'GPU_COUNT_min': 2}]}),
    ([],{})
]

match_ids_list = [
    # ((available_ids, patterns),[expected_results])
    ((['147e:1000', '147e:2016', '147e:3001', '147e:5002', '0483:2015', '1c7a:0603'],
      ['147e:100[0123]', '147e:20(15|16|20)', '147e:300[01]', '147e:500[23]', '0483:201[56]', '1c7a:0603']),
     ['147e:1000', '147e:2016', '147e:3001', '147e:5002', '0483:2015', '1c7a:0603']),

    ((['147e:1003','147e:2015', '147e:3000', '0483:2015', '1c7a:0603'],
      ['147e:100[0123]', '147e:20(15|16|20)', '147e:300[01]', '0483:201[56]', '1c7a:0603']),
     ['147e:1003','147e:2015', '147e:3000', '0483:2015', '1c7a:0603'])
]

@pytest.mark.parametrize("input_config", valid_config_list)
def test_validate_config(input_config):
    assert validate_config(input_config, "filename")

@pytest.mark.parametrize("input_config", invalid_config_list)
def test_invalidate_config(input_config):
    assert not validate_config(input_config, "filename")

@pytest.mark.parametrize("input_configs,expected_config", merge_configs_list)
def test_merge_configs(input_configs, expected_config):
    assert merge_configs(input_configs) == expected_config

def test_get_config(fs, monkeypatch):
    fs.create_file("/usr/lib/tuxedo-tomte-light/base.yaml")
    with open("/usr/lib/tuxedo-tomte-light/base.yaml", "w") as file:
        file.write("""
tuxedo-yt6801:
  - PCI: "1f0a:6801" # Motorcomm YT6801
tuxedo-fix-ath12k-standby:
  - PCI: "17cb:1107" # Qualcomm WCN785x
fingerprintreader:
  - USB:
    - "147e:100[0123]" # Upek Biometric Touchstrip Fingerprint Sensor (1000/1002) / TCS5B Fingerprint sensor (1001)
    - "147e:20(15|16|20)" # Upek Biometric Touchstrip Fingerprint Sensor (2016) / TouchChip Fingerprint Coprocessor (WBF advanced mode) (2020)
    - "147e:300[01]" # Upek TCS1C EIM/Cypress Fingerprint sensor (3000) / TCS1C EIM/STM32 Fingerprint sensor (3001)
    - "147e:500[23]" # Upek Unknown
    - "0483:201[56]" # STMicroelectronics Fingerprint Reader
    - "1c7a:0603" # LighTuning ES603 Swipe Fingerprint Sensor

"""
        )
    fs.create_file("/usr/lib/tuxedo-tomte-light/tuxedo-devices.yaml")
    with open("/usr/lib/tuxedo-tomte-light/tuxedo-devices.yaml", "w") as file:
        file.write("""
tuxedo-drivers:
tuxedo-fix-wifi-set-reg-domain:
  - PCI:
    - "14c3:0608" # MediaTek MT7921K / AMD RZ608
    - "14c3:0616" # MediaTek MT7922 / AMD RZ616

"""
        )
    fs.create_file("/usr/lib/tuxedo-tomte-light/tuxedo-os.yaml")
    with open("/usr/lib/tuxedo-tomte-light/tuxedo-os.yaml", "w") as file:
        file.write(
"""
tuxedo-dgpu-run:
  - GPU_COUNT_min: 2

"""
        )
    fs.create_file("/etc/os-release")
    with open("/etc/os-release", "w") as file:
        file.write("""
NAME="TUXEDO OS"
VERSION="24.04.1 LTS"
ID=tuxedo
ID_LIKE="ubuntu debian"
PRETTY_NAME="TUXEDO OS"
VERSION_ID="24.04"
HOME_URL="https://tuxedocomputers.com/"
SUPPORT_URL="https://support.tuxedocomputers.com/"
BUG_REPORT_URL="https://gitlab.com/tuxedocomputers/development/tuxedo_os/os"
PRIVACY_POLICY_URL="https://www.tuxedocomputers.com/en/Privacy-policy.tuxedo"
VERSION_CODENAME=noble
UBUNTU_CODENAME=noble
            """
        )
    # monkey patch is_tuxedo_device to always return True
    monkeypatch.setattr(src.tuxedo_tomte_light.config_parser, "is_tuxedo_device", lambda: True)
    returned_config = get_config()
    expected_config = {'tuxedo-yt6801': [{'PCI': '1f0a:6801'}], 'tuxedo-drivers': None, 'tuxedo-fix-ath12k-standby': [{'PCI': '17cb:1107'}],
     'fingerprintreader': [
         {'USB': ['147e:100[0123]', '147e:20(15|16|20)', '147e:300[01]', '147e:500[23]', '0483:201[56]', '1c7a:0603']}],
     'tuxedo-fix-wifi-set-reg-domain': [{'PCI': ['14c3:0608', '14c3:0616']}], 'tuxedo-dgpu-run': [{'GPU_COUNT_min': 2}]}
    assert returned_config == expected_config

@pytest.mark.parametrize("func_input,expected_results", match_ids_list)
def test_match_ids(func_input, expected_results):
    available_ids, patterns = func_input
    assert match_ids(available_ids, patterns) == expected_results

# TODO maybe parametrize this too
def test_get_os_ids(fs): # uses pytests pyfakefs to fake filesystem
    fs.create_file("/etc/os-release")
    with open("/etc/os-release", "w") as file:
        file.write("""
NAME="TUXEDO OS"
VERSION="24.04.1 LTS"
ID=tuxedo
ID_LIKE="ubuntu debian"
PRETTY_NAME="TUXEDO OS"
VERSION_ID="24.04"
HOME_URL="https://tuxedocomputers.com/"
SUPPORT_URL="https://support.tuxedocomputers.com/"
BUG_REPORT_URL="https://gitlab.com/tuxedocomputers/development/tuxedo_os/os"
PRIVACY_POLICY_URL="https://www.tuxedocomputers.com/en/Privacy-policy.tuxedo"
VERSION_CODENAME=noble
UBUNTU_CODENAME=noble
            """
        )
    assert get_os_ids() == {"tuxedo", "ubuntu", "debian"}
    arch_os_release = """
NAME = "Arch Linux"
PRETTY_NAME = "Arch Linux"
ID = arch
BUILD_ID = rolling
ANSI_COLOR = "38;2;23;147;209"
HOME_URL = "https://archlinux.org/"
DOCUMENTATION_URL = "https://wiki.archlinux.org/"
SUPPORT_URL = "https://bbs.archlinux.org/"
BUG_REPORT_URL = "https://gitlab.archlinux.org/groups/archlinux/-/issues"
PRIVACY_POLICY_URL = "https://terms.archlinux.org/docs/privacy-policy/"
LOGO = archlinux - logo
            """
    with open("/etc/os-release", "w") as file:
        file.write(arch_os_release)
    assert get_os_ids() != {"tuxedo", "ubuntu", "debian"}
    assert get_os_ids() == {"arch"}

def test_resolve_mapping(fs): # uses pytests pyfakefs to fake filesystem
    fs.create_file("/etc/os-release")
    with open("/etc/os-release", "w") as file:
        file.write(
"""
NAME="TUXEDO OS"
VERSION="24.04.1 LTS"
ID=tuxedo
ID_LIKE="ubuntu debian"
PRETTY_NAME="TUXEDO OS"
VERSION_ID="24.04"
HOME_URL="https://tuxedocomputers.com/"
SUPPORT_URL="https://support.tuxedocomputers.com/"
BUG_REPORT_URL="https://gitlab.com/tuxedocomputers/development/tuxedo_os/os"
PRIVACY_POLICY_URL="https://www.tuxedocomputers.com/en/Privacy-policy.tuxedo"
VERSION_CODENAME=noble
UBUNTU_CODENAME=noble
"""
        )
    fs.create_file("/usr/lib/tuxedo-tomte-light/package_mapper.yaml")
    with open("/usr/lib/tuxedo-tomte-light/package_mapper.yaml", "w") as file:
        file.write(
"""
fingerprintreader:
  ubuntu: "libpam-fprintd"
"""
        )
    packages = {"fingerprintreader", "another-package", "third-package"}
    new_packages = resolve_mapping(packages)
    assert "libpam-fprintd" in new_packages
    assert "fingerprintreader" not in new_packages
    # add second test to see that in non ubuntu it doesn't do that
    with open("/etc/os-release", "w") as file:
        file.write("")
    new_packages = resolve_mapping(packages)
    assert "libpam-fprintd" not in new_packages
    assert "fingerprintreader" in new_packages

def test_get_hardware_config_dependency_packages(fs, monkeypatch):
    config = {'tuxedo-yt6801': [{'PCI': '1f0a:6801'}], 'tuxedo-drivers': None,
              'tuxedo-fix-ath12k-standby': [{'PCI': '17cb:1107'}],
              'fingerprintreader': [
         {'USB': ['147e:100[0123]', '147e:20(15|16|20)', '147e:300[01]', '147e:500[23]', '0483:201[56]', '1c7a:0603']}],
     'tuxedo-fix-wifi-set-reg-domain': [{'PCI': ['14c3:0608', '14c3:0616']}], 'tuxedo-dgpu-run': [{'GPU_COUNT_min': 2}]}
    hardware_specs = {
         'PCI': ['1f0a:6801', '17cb:1107'],
         'USB': ['147e:3001'],
         'GPU_COUNT': 2
      }
    monkeypatch.setattr(src.tuxedo_tomte_light.package_mapper, "get_config", lambda: config)
    monkeypatch.setattr(src.tuxedo_tomte_light.package_mapper, "get_hardware_specs", lambda: hardware_specs)
    # also monkeypatch everything for resolve mapper
    fs.create_file("/etc/os-release")
    with open("/etc/os-release", "w") as file:
        file.write(
            """
            NAME="TUXEDO OS"
            VERSION="24.04.1 LTS"
            ID=tuxedo
            ID_LIKE="ubuntu debian"
            PRETTY_NAME="TUXEDO OS"
            VERSION_ID="24.04"
            HOME_URL="https://tuxedocomputers.com/"
            SUPPORT_URL="https://support.tuxedocomputers.com/"
            BUG_REPORT_URL="https://gitlab.com/tuxedocomputers/development/tuxedo_os/os"
            PRIVACY_POLICY_URL="https://www.tuxedocomputers.com/en/Privacy-policy.tuxedo"
            VERSION_CODENAME=noble
            UBUNTU_CODENAME=noble
            """
        )
    fs.create_file("/usr/lib/tuxedo-tomte-light/package_mapper.yaml")
    with open("/usr/lib/tuxedo-tomte-light/package_mapper.yaml", "w") as file:
        file.write(
            """
            fingerprintreader:
              ubuntu: "libpam-fprintd"
            """
        )
    expected_packages = {'tuxedo-yt6801', 'tuxedo-fix-ath12k-standby', 'tuxedo-drivers',
                         'libpam-fprintd', 'tuxedo-dgpu-run'}
    returned_packages = get_hardware_config_dependend_packages()
    assert expected_packages == returned_packages
