from pathlib import Path

import src
from src.tuxedo_tomte_light.board_manager import get_dmi_id_string, str_to_debian_package_name, get_dmi_dependend_packages
from src.tuxedo_tomte_light.file_operations import read_file, load_yaml_file
from src.tuxedo_tomte_light.hardware_parser import count_gpu_devices, get_hardware_specs, \
    get_pci_devices, get_usb_devices
from src.tuxedo_tomte_light.package_manager import PackageManager
from src.tuxedo_tomte_light.tuxedo_detection import is_notebook, is_tuxedo_os, is_tuxedo_device
import pyfakefs
import pytest

# from board_manager
dmi_strings_list = [("key1", "value1"),("boardname", "cool_mainboardxyz")]
@pytest.mark.parametrize("dmi_id_string,expected_value", dmi_strings_list)
def test_get_dmi_id_string(dmi_id_string, expected_value, fs):
    fs.create_file(f"/sys/class/dmi/id/{dmi_id_string}", contents=expected_value)
    result = get_dmi_id_string(dmi_id_string)
    assert result == expected_value.strip()


def test_get_dmi_id_string_empty(fs):
    result = get_dmi_id_string("class")
    assert result is None


@pytest.mark.parametrize("dmi_string,package_name", [("Micro-Star International Co., Ltd", "micro-star-international-co.-ltd"), ("B450 TOMAHAWK MAX II (MS-7C02)","b450-tomahawk-max-ii-ms-7c02"), ("NB01","nb01"), ("NS50_70MU","ns50-70mu")])
def test_str_to_debian_package_name(dmi_string, package_name):
    assert str_to_debian_package_name(dmi_string) == package_name


def test_get_dmi_dependend_packages(fs):
    class MockPackageManager(PackageManager):
        def install(self, package):
            return True
        def update(self):
            return True
        def package_exists(self, package):
            return True
    pm = MockPackageManager()
    board_name = "NS50_70MU"
    board_vendor = "NB01"
    fs.create_file("/sys/class/dmi/id/board_name", contents=board_name)
    fs.create_file("/sys/class/dmi/id/board_vendor", contents=board_vendor)
    result = get_dmi_dependend_packages(pm)
    expected = ["tuxedo-board-vendor-nb01", "tuxedo-board-ns50-70mu"]
    assert sorted(list(result)) == sorted(expected)


# from file_operations
def test_read_file(fs):
    path = Path("test")
    contents = read_file(path)
    assert contents is None
    fs.create_file(path)
    contents = read_file(path)
    assert contents is ''
    with open(path, "w") as file:
        file.write("test contents")
    contents = read_file(path)
    assert contents == "test contents"


def test_load_yaml_file(fs):
    path = "yaml_path"
    result = load_yaml_file(path)
    assert result == {}
    fs.create_dir(path)
    assert result == {}
    path = "yaml_path/file.yaml"
    fs.create_file(path)
    result = load_yaml_file(path)
    assert result == {}
    yaml_contents="""
item1:
  - listitem1
item2:
  - listitem21
  - listitem22
"""
    with open(path, "w") as file:
        file.write(yaml_contents)
    result = load_yaml_file(path)
    assert result == {"item1":["listitem1"],"item2":["listitem21", "listitem22"]}


pci_usb_devices_mock_data_list = [
    ("hardware_infodump_GXxHRXx_ibp15g9_amd", (["1022:14e8", "1022:14e9", "1022:14ea", "1022:14ed",
    "1022:14ea", "1022:14ee", "1022:14ee", "1022:14ee", "1022:14ea", "1022:14ea", "1022:14ef", "1022:14ea",
    "1022:14eb", "1022:14eb", "1022:14eb", "1022:790b", "1022:790e", "1022:14f0", "1022:14f1", "1022:14f2",
    "1022:14f3", "1022:14f4", "1022:14f5", "1022:14f6", "1022:14f7", "144d:a80d", "1f0a:6801", "8086:2725",
    "17a0:9767", "1002:1900", "1002:1640", "1022:15c7", "1022:15b9", "1022:15ba", "1022:15e2", "1022:15e3",
    "1022:14ec", "1022:1502", "1022:14ec", "1022:15c0", "1022:15c1", "1022:1669"
                                                ], [
        "05e3:0610", "ffff:5678", "8087:0032", "05e3:0620", "2b7e:c858", "1d6b:0002", "1d6b:0003", "1d6b:0002",
        "1d6b:0003", "1d6b:0002", "1d6b:0003", "1d6b:0002", "1d6b:0003"
    ])),
    ("hardware_infodump_GMxXGxx_polaris15g5amd", ([
        "1022:14e8", "1022:14e9", "1022:14ea", "1022:14ed", "1022:14ea", "1022:14ee", "1022:14ee", "1022:14ee",
        "1022:14ee", "1022:14ea", "1022:14ea", "1022:14ea", "1022:14eb", "1022:14eb", "1022:14eb", "1022:790b",
        "1022:790e", "1022:14f0", "1022:14f1", "1022:14f2", "1022:14f3", "1022:14f4", "1022:14f5", "1022:14f6",
        "1022:14f7", "10de:28a1", "10de:22be", "10ec:8168", "14c3:0608", "c0a9:540a", "1002:15bf", "1002:1640",
        "1022:15c7", "1022:15b9", "1022:15ba", "1022:15e2", "1022:15e3", "1022:14ec", "1022:1502", "1022:14ec",
        "1022:15c0", "1022:15c1"
                                                  ], [
        "ffff:5678", "04f2:b642", "0e8d:0608", "05e3:0608", "1d6b:0002", "1d6b:0003", "1d6b:0002", "1d6b:0003",
        "1d6b:0002", "1d6b:0003", "1d6b:0002", "1d6b:0003"
    ]))
]
# from hardware_parser f"{vendor}:{device}"
@pytest.mark.parametrize("dump_name,expected_lists", pci_usb_devices_mock_data_list)
def test_get_pci_usb_devices(fs, dump_name, expected_lists): # (die anderen beiden funktionen geben nur substrings davon wieder)
    fs.add_real_directory(source_path="src/tests/mock_data/hardware_dumps/"+dump_name+"/sys", target_path="/sys")
    expected_pci, expected_usb = expected_lists
    pci_ids = get_pci_devices()
    usb_ids = get_usb_devices()
    assert sorted(pci_ids) == sorted(expected_pci)
    assert sorted(usb_ids) == sorted(expected_usb)


dump_names_count=[("hardware_infodump_GXxHRXx_ibp15g9_amd", 1), ("hardware_infodump_GMxXGxx_polaris15g5amd", 2)]
@pytest.mark.parametrize("dump_name, count", dump_names_count)
def test_count_gpu_devices(fs, dump_name, count):
    fs.add_real_directory(source_path="src/tests/mock_data/hardware_dumps/" + dump_name + "/sys", target_path="/sys")
    c = count_gpu_devices()
    # assert c == count
    # TODO add class to mock-data


pci_usb_devices_mock_data_list_count = [
    ("hardware_infodump_GXxHRXx_ibp15g9_amd", (["1022:14e8", "1022:14e9", "1022:14ea", "1022:14ed",
    "1022:14ea", "1022:14ee", "1022:14ee", "1022:14ee", "1022:14ea", "1022:14ea", "1022:14ef", "1022:14ea",
    "1022:14eb", "1022:14eb", "1022:14eb", "1022:790b", "1022:790e", "1022:14f0", "1022:14f1", "1022:14f2",
    "1022:14f3", "1022:14f4", "1022:14f5", "1022:14f6", "1022:14f7", "144d:a80d", "1f0a:6801", "8086:2725",
    "17a0:9767", "1002:1900", "1002:1640", "1022:15c7", "1022:15b9", "1022:15ba", "1022:15e2", "1022:15e3",
    "1022:14ec", "1022:1502", "1022:14ec", "1022:15c0", "1022:15c1", "1022:1669"
                                                ], [
        "05e3:0610", "ffff:5678", "8087:0032", "05e3:0620", "2b7e:c858", "1d6b:0002", "1d6b:0003", "1d6b:0002",
        "1d6b:0003", "1d6b:0002", "1d6b:0003", "1d6b:0002", "1d6b:0003"
    ], 1)),
    ("hardware_infodump_GMxXGxx_polaris15g5amd", ([
        "1022:14e8", "1022:14e9", "1022:14ea", "1022:14ed", "1022:14ea", "1022:14ee", "1022:14ee", "1022:14ee",
        "1022:14ee", "1022:14ea", "1022:14ea", "1022:14ea", "1022:14eb", "1022:14eb", "1022:14eb", "1022:790b",
        "1022:790e", "1022:14f0", "1022:14f1", "1022:14f2", "1022:14f3", "1022:14f4", "1022:14f5", "1022:14f6",
        "1022:14f7", "10de:28a1", "10de:22be", "10ec:8168", "14c3:0608", "c0a9:540a", "1002:15bf", "1002:1640",
        "1022:15c7", "1022:15b9", "1022:15ba", "1022:15e2", "1022:15e3", "1022:14ec", "1022:1502", "1022:14ec",
        "1022:15c0", "1022:15c1"
                                                  ], [
        "ffff:5678", "04f2:b642", "0e8d:0608", "05e3:0608", "1d6b:0002", "1d6b:0003", "1d6b:0002", "1d6b:0003",
        "1d6b:0002", "1d6b:0003", "1d6b:0002", "1d6b:0003"
    ], 1))
]
@pytest.mark.parametrize("dump_name,expected_lists", pci_usb_devices_mock_data_list_count)
def test_get_hardware_specs(dump_name, expected_lists, fs):
    fs.add_real_directory(source_path="src/tests/mock_data/hardware_dumps/" + dump_name + "/sys", target_path="/sys")
    hardware_specs= get_hardware_specs()
    expected_pci, expected_usb, expected_count = expected_lists
    expected_dict = {
        'PCI': expected_pci,
        'USB': expected_usb,
        'GPU_COUNT': expected_count
    }
    assert sorted(hardware_specs['PCI']) == sorted(expected_dict['PCI'])
    assert sorted(hardware_specs['USB']) == sorted(expected_dict['USB'])


# from tuxedo_detection
def test_is_notebook(monkeypatch):
    dmidecode_output=""
    with open("src/tests/mock_data/hardware_dumps/dmidecode/dmidecode_nb01.txt") as file:
        dmidecode_output = file.read()
    monkeypatch.setattr(
        src.tuxedo_tomte_light.tuxedo_detection.subprocess,
        "check_output",
        lambda *args, **kwargs: dmidecode_output
    )

    assert is_notebook()


# fakefs wie in anderen unit tests bei config
def test_is_tuxedo_os(fs):
    fs.create_file("/etc/os-release")
    with open("/etc/os-release", "w") as file:
        file.write("""
NAME="TUXEDO OS"
VERSION="24.04.1 LTS"
ID=tuxedo
ID_LIKE="ubuntu debian"
PRETTY_NAME="TUXEDO OS"
VERSION_ID="24.04"
HOME_URL="https://tuxedocomputers.com/"
SUPPORT_URL="https://support.tuxedocomputers.com/"
BUG_REPORT_URL="https://gitlab.com/tuxedocomputers/development/tuxedo_os/os"
PRIVACY_POLICY_URL="https://www.tuxedocomputers.com/en/Privacy-policy.tuxedo"
VERSION_CODENAME=noble
UBUNTU_CODENAME=noble
            """
        )
    assert is_tuxedo_os()
    arch_os_release = """
NAME = "Arch Linux"
PRETTY_NAME = "Arch Linux"
ID = arch
BUILD_ID = rolling
ANSI_COLOR = "38;2;23;147;209"
HOME_URL = "https://archlinux.org/"
DOCUMENTATION_URL = "https://wiki.archlinux.org/"
SUPPORT_URL = "https://bbs.archlinux.org/"
BUG_REPORT_URL = "https://gitlab.archlinux.org/groups/archlinux/-/issues"
PRIVACY_POLICY_URL = "https://terms.archlinux.org/docs/privacy-policy/"
LOGO = archlinux - logo
            """
    with open("/etc/os-release", "w") as file:
        file.write(arch_os_release)
    assert not is_tuxedo_os()


#parametrize mit fakefs für dmi strings und dann true/false expected
# dump_names_is_tuxedo=[("hardware_infodump_GXxHRXx_ibp15g9_amd", True), ("hardware_infodump_GMxXGxx_polaris15g5amd", True)]
# @pytest.mark.parametrize("dump_name,expected_tuxedo", dump_names_is_tuxedo)
# def test_is_tuxedo_device(fs, dump_name, expected_tuxedo):
#     fs.add_real_directory(source_path="src/tests/mock_data/hardware_dumps/" + dump_name + "/sys/class/dmi/id/sys_vendor", target_path="/sys/devices/virtual/dmi/id/sys_vendor")
#     fs.add_real_directory(source_path="src/tests/mock_data/hardware_dumps/" + dump_name + "/sys/class/dmi/id/chassis_vendor", target_path="/sys/devices/virtual/dmi/id/chassis_vendor")
#     fs.add_real_directory(source_path="src/tests/mock_data/hardware_dumps/" + dump_name + "/sys/class/dmi/id/board_vendor", target_path="/sys/devices/virtual/dmi/id/board_vendor")
#     fs.add_real_directory(source_path="src/tests/mock_data/hardware_dumps/" + dump_name + "/sys/class/dmi/id/product_name", target_path="/sys/devices/virtual/dmi/id/product_name")
#     fs.add_real_directory(source_path="src/tests/mock_data/hardware_dumps/" + dump_name + "/sys/class/dmi/id/board_name", target_path="/sys/devices/virtual/dmi/id/board_name")
#     assert is_tuxedo_device() == expected_tuxedo # TODO fix mock data and then readd this

def test_is_tuxedo_device(fs):
    path = "/sys/devices/virtual/dmi/id/sys_vendor"
    fs.create_file(path)
    assert not is_tuxedo_device()
    with open(path, "w") as file:
        file.write("tuxedo")
    assert is_tuxedo_device()
    path = "/sys/devices/virtual/dmi/id/chassis_vendor"
    fs.create_file(path)
    with open(path, "w") as file:
        file.write("tuxedo")
    assert is_tuxedo_device()

    with open(path, "w") as file:
        file.write("TUXEDO")
    assert is_tuxedo_device()
    path = "/sys/devices/virtual/dmi/id/product_name"
    fs.create_file(path)
    with open(path, "w") as file:
        file.write("tuxedo")
    assert is_tuxedo_device()
    path = "/sys/devices/virtual/dmi/id/board_name"
    fs.create_file(path)
    with open(path, "w") as file:
        file.write("tuxedo")
    assert is_tuxedo_device()
