import argparse

from .update_hardware_information import update_hardware_information
from .fatal_checks import check_fatals
from .board_manager import get_dmi_dependend_packages
from .logging import logger
from .package_manager import detect_package_manager
from .package_mapper import (
    get_hardware_config_dependend_packages,
    resolve_mapping,
)
from .state_manager import mark_installed, reset_installed_state


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "--reset", action="store_true", help="Delete all stored installation flags"
    )
    parser.add_argument(
        "--force",
        action="store_true",
        help="Delete all stored installation flags and start",
    )
    args = parser.parse_args()

    check_fatals() # TODO move network check to after reset

    if args.reset or args.force:
        logger.info("Resetting installation state...")
        if not reset_installed_state():
            return
        if args.reset:
            return

    logger.info("Updating Hardware Information…")
    update_hardware_information()
    logger.info("Updated Hardware Information")

    logger.info("Evaluating configuration rules…")
    pkgs = get_hardware_config_dependend_packages()
    logger.info(f"Packages required by hardware: {sorted(pkgs)}")

    logger.info("Applying OS-dependent package mapping…")
    hw_pkgs = resolve_mapping(pkgs)
    logger.info(f"Final package list after mapping: {sorted(hw_pkgs)}")

    pm = detect_package_manager()
    if not pm:
        logger.fatal("Cannot find usable package manager. Exiting.")
        return
    logger.info(f"Using package manager: {pm.__class__.__name__}")

    dmi_pkgs = get_dmi_dependend_packages(pm)

    pkgs = list()
    for pkg in set(list(hw_pkgs) + list(dmi_pkgs)):
        if pm.package_exists(pkg):
            pkgs.append(pkg)
        else:
            logger.warning(f"No installation candidate for {pkg} in repositories")

    if not pkgs:
        logger.info("All packages already installed.")
    else:
        logger.info(f"Installing {len(pkgs)} packages in one call:")
        logger.info("  " + ", ".join(pkgs))

        ok = pm.install(list(pkgs))

        if ok:
            for pkg in hw_pkgs:
                if mark_installed(pkg):
                    logger.info(f"[OK] Installed {pkg}")
                else:
                    logger.error(f"[FAILED] to mark {pkg} as installed")
        else:
            logger.error(
                "[FAILED] Package manager reported failure during batch install"
            )

    logger.info("All done.")


if __name__ == "__main__":
    main()
