import re

from .config_parser import get_config, load_yaml_file
from .hardware_parser import get_hardware_specs
from .state_manager import is_already_installed
from .logging import logger

def match_ids(available_ids, patterns):
    logger.debug("matching available ids in patterns")
    logger.debug("ids:")
    logger.debug(available_ids)
    logger.debug("pattersn:")
    logger.debug(patterns)
    matched = []
    for pat in patterns:
        regex = re.compile(pat)
        for dev_id in available_ids:
            if regex.fullmatch(dev_id):
                matched.append(dev_id)
    logger.debug("matches found:")
    logger.debug(matched)
    return matched


def get_hardware_config_dependend_packages() -> set[str]:
    """Return set of hardware dependent packages based on PCI/USB matches."""
    packages = set()
    hardware_specs = get_hardware_specs()
    config = get_config()
    logger.debug("getting config for hardware:")
    logger.debug(hardware_specs)
    logger.debug("config:")
    logger.debug(config)
    for package, conditions in config.items():
        if not conditions:
            packages.add(package)
            continue

        for cond in conditions:
            add_package = True
            if "PCI" in cond:
                patterns = cond["PCI"]
                if not isinstance(patterns, list):
                    patterns = [patterns]
                if not match_ids(hardware_specs["PCI"], patterns):
                    add_package = False
            if "USB" in cond:
                patterns = cond["USB"]
                if not isinstance(patterns, list):
                    patterns = [patterns]
                if not match_ids(hardware_specs["USB"], patterns):
                    add_package = False
            if "GPU_COUNT_min" in cond:
                if int(cond["GPU_COUNT_min"]) > int(hardware_specs["GPU_COUNT"]):
                    add_package = False

            if add_package:
                packages.add(package)

    logger.debug("packages before mapping:")
    logger.debug(packages)

    resolved_packages = resolve_mapping(packages)
    logger.debug("resolved packages:")
    logger.debug(resolved_packages)
    return resolved_packages


def get_os_ids() -> set[str]:
    ids = set()
    try:
        with open("/etc/os-release", "r") as f:
            for line in f:
                if "=" in line:
                    key, val = line.strip().split("=")
                    key = key.strip()
                    val = val.strip()
                    val = val.strip('"')
                    if key == "ID":
                        ids.add(val)
                    elif key == "ID_LIKE":
                        ids.update(val.split())
    except FileNotFoundError:
        logger.error("No OS-Release file found!")
        pass
    return ids


def resolve_mapping(packages: set[str]) -> set[str]:
    os_identifiers = get_os_ids()
    package_map = load_yaml_file("/usr/lib/tuxedo-tomte-light/package_mapper.yaml")
    logger.debug("package map:")
    logger.debug(package_map)
    packages_final = set()

    for package in packages:
        mapped_value = package  # default: package stays unchanged
        if package in package_map:
            # iterate through mapping entries of this package
            for os_key, replacement in package_map[package].items():
                if os_key in os_identifiers:
                    mapped_value = replacement
                    break
        if not is_already_installed(mapped_value):
            packages_final.add(mapped_value)
    logger.debug("final packages:")
    logger.debug(packages_final)
    return packages_final
