import logging

from .board_manager import get_dmi_dependend_packages
from .fatal_handling import check_fatals, tomte_light_exit
from .logging import setup_logging
from .package_manager import detect_package_manager
from .package_mapper import (
    get_hardware_config_dependend_packages,
    resolve_mapping,
)
from .state_manager import mark_installed
from .update_hardware_information import update_hardware_information
from .args_management import parse_args, execute_flags_and_check_return_necessary

logger = logging.getLogger(__name__)


def main() -> None:
    setup_logging()

    args = parse_args()
    check_fatals()  # TODO move network check to after reset
    if execute_flags_and_check_return_necessary(args):
        return

    logger.info("Updating Hardware Information…")
    update_hardware_information()
    logger.info("Updated Hardware Information")

    logger.info("Evaluating configuration rules…")
    pkgs: set[str] = get_hardware_config_dependend_packages()
    logger.info(f"Packages required by hardware: {sorted(pkgs)}")

    logger.info("Applying OS-dependent package mapping…")
    hw_pkgs = resolve_mapping(pkgs)
    logger.info(f"Final package list after mapping: {sorted(hw_pkgs)}")

    pm = detect_package_manager()
    if not pm:
        logger.fatal("Cannot find usable package manager. Exiting.")
        return
    logger.info(f"Using package manager: {pm.__class__.__name__}")

    dmi_pkgs = get_dmi_dependend_packages(pm)

    pkg_list: list[str] = []
    for pkg in set(list(hw_pkgs) + list(dmi_pkgs)):
        if pm.package_exists(pkg):
            pkg_list.append(pkg)
        else:
            logger.warning(f"No installation candidate for {pkg} in repositories")

    if not pkg_list:
        logger.info("All packages already installed.")
    else:
        logger.info(f"Installing {len(pkg_list)} packages in one call:")
        logger.info("  " + ", ".join(pkg_list))

        ok = pm.install(list(pkg_list))

        if ok:
            for pkg in hw_pkgs:
                if mark_installed(pkg):
                    logger.info(f"[OK] Installed {pkg}")
                else:
                    logger.error(f"[FAILED] to mark {pkg} as installed")
        else:
            logger.error(
                "[FAILED] Package manager reported failure during batch install"
            )

    logger.info("All done.")


if __name__ == "__main__":
    main()
    tomte_light_exit(0)
