import logging
import os
import shutil
import subprocess
from abc import ABC, abstractmethod
from typing import override

logger = logging.getLogger(__name__)
packages_logger = logging.getLogger(f"{__name__.split('.', 1)[0]}.packages")


class PackageManager(ABC):
    BIN: str  # Name des Binaries, muss in Subklassen angegeben werden

    @staticmethod
    def _run(args: list[str]) -> tuple[int, str, str]:
        """Executes a system command with full logging and returns (rc, stdout, stderr)."""
        logger.debug(f"Executing command: {' '.join(args)}")

        proc = subprocess.run(
            args,
            check=False,
            capture_output=True,
            text=True,
            env={
                **os.environ,
                "LANG": "C",
                "LC_ALL": "C",
            },
        )

        logger.debug(f"Return code: {proc.returncode}")
        if proc.stdout:
            logger.debug(f"Stdout:\n{proc.stdout.rstrip()}")
        if proc.stderr:
            logger.debug(f"Stderr:\n{proc.stderr.rstrip()}")

        return proc.returncode, proc.stdout, proc.stderr

    @staticmethod
    def _run_checked(
        args: list[str],
        success_codes: list[int] = [0],  # noqa: B006  # pyright: ignore[reportCallInDefaultInitializer]
    ) -> tuple[bool, str]:
        """Runs a command and compares the return code to a given list of return codes treated as success."""
        rc, out, _err = PackageManager._run(args)
        success = True
        if rc not in success_codes:
            success = False
        return success, out

    @staticmethod
    def _normalize_packages(packages: list[str] | str) -> list[str]:
        if isinstance(packages, str):
            logger.debug(
                "Received a single string instead of list[str]. Autocorrecting..."
            )
            return packages.split()
        return packages

    # ---- Abstract API ----

    @abstractmethod
    def install(self, packages: list[str] | str) -> bool: ...
    @abstractmethod
    def update(self) -> bool: ...
    @abstractmethod
    def package_exists(self, pkgname: str) -> bool: ...


class AptManager(PackageManager):
    BIN: str = "apt-get"

    @override
    def install(self, packages: list[str] | str) -> bool:
        packages = self._normalize_packages(packages)
        packages_logger.info(f"APT installing: {packages}")
        logger.info(f"Installing via APT (with recommends): {packages}")
        return self._run_checked(
            [self.BIN, "install", "-y", "--install-recommends", *packages]
        )[0]

    @override
    def update(self) -> bool:
        logger.info("Updating APT package lists.")
        return self._run_checked([self.BIN, "update"])[0]

    @override
    def package_exists(self, pkgname: str) -> bool:
        logger.debug(f"Checking if APT package exists: {pkgname}")
        _success, out = self._run_checked(["apt-cache", "policy", pkgname])

        for line in out.splitlines():
            if line.strip().startswith("Candidate:") and "(none)" not in line:
                return True

        return False


class DnfManager(PackageManager):
    BIN: str = "dnf"

    @override
    def install(self, packages: list[str] | str) -> bool:
        packages = self._normalize_packages(packages)
        packages_logger.info(f"DNF installing: {packages}")
        logger.info(f"Installing via DNF: {packages}")
        return self._run_checked([self.BIN, "-y", "install", *packages])[0]

    @override
    def update(self) -> bool:
        logger.info("Checking for DNF updates.")

        # DNF returns:
        #   0   = no updates available (success)
        #   100 = updates available (still success)
        #   else = error
        return self._run_checked([self.BIN, "check-update"], success_codes=[0, 100])[0]

    @override
    def package_exists(self, pkgname: str) -> bool:
        """
        Reliable DNF package existence check.

        1. Check if package is installed (rpm -q)
        2. Check if package is available (dnf repoquery)
        3. Fallback: dnf list available
        """
        logger.debug(f"Checking if DNF package exists: {pkgname}")
        # Step 1: Installed?
        if self._run_checked(["rpm", "-q", pkgname])[0]:
            return True

        # Step 2: Available? (DNF's official way)
        # repoquery returns 0 only if package exists
        success, out = self._run_checked(["dnf", "-q", "repoquery", pkgname])
        if success and pkgname in out:
            return True

        # Step 3: Fallback for minimal DNF builds
        success, out = self._run_checked(["dnf", "-q", "list", "available", pkgname])
        return "Available Packages" in out and pkgname in out


class PacmanManager(PackageManager):
    BIN: str = "pacman"

    @override
    def install(self, packages: list[str] | str) -> bool:
        packages = self._normalize_packages(packages)
        packages_logger.info(f"pacman installing: {packages}")
        logger.info(f"Installing via Pacman: {packages}")
        return self._run_checked(
            [self.BIN, "-S", "--needed", "--noconfirm", *packages]
        )[0]

    @override
    def update(self) -> bool:
        logger.info("Refreshing Pacman repositories.")
        return self._run_checked([self.BIN, "-Sy"])[0]

    @override
    def package_exists(self, pkgname: str) -> bool:
        logger.debug(f"Checking if pacman package exists: {pkgname}")
        (
            _success,
            out,
        ) = self._run_checked(["pacman", "-Si", pkgname])
        return "Repository" in out


class ZypperManager(PackageManager):
    BIN: str = "zypper"

    @override
    def install(self, packages: list[str] | str) -> bool:
        packages = self._normalize_packages(packages)
        packages_logger.info(f"zyper installing: {packages}")
        logger.info(f"Installing via Zypper (with recommends): {packages}")
        return self._run_checked(
            [self.BIN, "--non-interactive", "install", "--recommends", *packages]
        )[0]

    @override
    def update(self) -> bool:
        logger.info("Refreshing Zypper repositories.")
        return self._run_checked([self.BIN, "--non-interactive", "refresh"])[0]

    @override
    def package_exists(self, pkgname: str) -> bool:
        logger.debug(f"Checking if Zyper package exists: {pkgname}")
        (
            _success,
            out,
        ) = self._run_checked(["zypper", "search", "--match-exact", pkgname])
        return pkgname in out


def detect_package_manager() -> PackageManager | None:
    """
    Automatically detects the active Linux package manager by scanning all
    PackageManager subclasses and checking their BIN attribute.
    """

    subclasses = PackageManager.__subclasses__()

    priority = ["apt-get", "dnf", "pacman", "zypper"]

    subclasses.sort(
        key=lambda class_name: priority.index(class_name.BIN)
        if getattr(class_name, "BIN", None) in priority
        else 999
    )

    for cls in subclasses:
        binary = getattr(cls, "BIN", None)
        if binary and shutil.which(binary):
            pm = cls()
            logger.info(f"Detected package manager: {cls.__name__} (BIN={binary})")
            return pm

    logger.error("Kein unterstützter Paketmanager gefunden.")
    return None
