import re
from pathlib import Path
from .package_manager import PackageManager
import logging

logger = logging.getLogger(__name__)


def get_dmi_id_string(dmi_id_string: str) -> str:
    try:
        with Path(f"/sys/class/dmi/id/{dmi_id_string}").open() as f:
            return f.read().strip()
    except FileNotFoundError:
        logger.error(f"dmi string {dmi_id_string} not found")
        raise


def str_to_debian_package_name(name: str) -> str:
    """
    Wandelt einen beliebigen String in einen Debian-Paketnamen um.
    Regeln:
    - alles in Kleinbuchstaben
    - Leerzeichen und unerlaubte Zeichen durch '-'
    - nur a-z, 0-9, '+', '-', '.' erlaubt
    - aufeinanderfolgende '-' werden zusammengefasst
    - führende/trailing '-' werden entfernt
    """
    logger.debug(f"trying to convert: {name}")
    name = name.lower()
    # erlaubte Zeichen: a-z, 0-9, +, -, .
    # alles andere durch '-'
    name = re.sub(r"[^a-z0-9+\.-]+", "-", name)
    # aufeinanderfolgende '-' zu einem zusammenfassen
    name = re.sub(r"-{2,}", "-", name)
    # führende/trailing '-' entfernen
    name = name.strip("-")
    logger.debug(f"resulting package name: {name}")
    return name


def get_dmi_dependend_packages(_pm: PackageManager) -> set[str]:
    """Return set of board/vendor dependent packages that exist."""
    packages: set[str] = set()

    board_name = str_to_debian_package_name(get_dmi_id_string("board_name"))
    if board_name:
        packages.add(f"tuxedo-board-{board_name}")

    board_vendor = str_to_debian_package_name(get_dmi_id_string("board_vendor"))
    if board_vendor:
        packages.add(f"tuxedo-board-vendor-{board_vendor}")

    return packages
