import logging
import re
from pathlib import Path

from .config_parser import ConfigEntry, get_config
from .file_operations import load_yaml_file
from .hardware_parser import HardwareSpecs
from .state_manager import is_already_installed

logger = logging.getLogger(__name__)


def match_ids(available_ids: list[str], patterns: list[str]) -> list[str]:
    logger.debug("matching available ids in patterns")
    logger.debug("ids:")
    logger.debug(available_ids)
    logger.debug("pattersn:")
    logger.debug(patterns)
    matched: list[str] = []
    for pat in patterns:
        regex = re.compile(pat)
        matched.extend(dev_id for dev_id in available_ids if regex.fullmatch(dev_id))
    logger.debug("matches found:")
    logger.debug(matched)
    return matched


def config_match(hardware_specs: HardwareSpecs, entry: ConfigEntry) -> bool:
    if entry.pci is not None:
        patterns = entry.pci
        if not match_ids(hardware_specs.pci, patterns):
            return False
    if entry.usb is not None:
        patterns = entry.usb
        if not match_ids(hardware_specs.usb, patterns):
            return False
    if (
        entry.gpu_count_min is not None
        and entry.gpu_count_min > hardware_specs.gpu_count
    ):
        return False

    for not_rule in entry.not_:
        # If at least one not-rule matches, it doesn't match
        if config_match(hardware_specs, not_rule):
            return False

    for any_rule in entry.any:
        # If at least one any-rule matches, it matches
        if config_match(hardware_specs, any_rule):
            return True

    return True


def get_hardware_config_dependend_packages() -> set[str]:
    """Return set of hardware dependent packages based on PCI/USB matches."""
    packages: set[str] = set()
    hardware_specs = HardwareSpecs()
    config = get_config()
    logger.debug("getting config for hardware:")
    logger.debug(hardware_specs)
    logger.debug("config:")
    logger.debug(config)
    for package, cond in config.root.items():
        if not cond or config_match(hardware_specs, cond):
            packages.add(package)

    logger.debug("packages before mapping:")
    logger.debug(packages)

    resolved_packages = resolve_mapping(packages)
    logger.debug("resolved packages:")
    logger.debug(resolved_packages)
    return resolved_packages


def get_os_ids() -> set[str]:
    ids: set[str] = set()
    try:
        with Path("/etc/os-release").open() as f:
            for line in f:
                if "=" in line:
                    key, val = line.strip().split("=")
                    key = key.strip()
                    val = val.strip()
                    val = val.strip('"')
                    if key == "ID":
                        ids.add(val)
                    elif key == "ID_LIKE":
                        ids.update(val.split())
    except FileNotFoundError:
        logger.error("No OS-Release file found!")
    return ids


def resolve_mapping(packages: set[str]) -> set[str]:
    os_identifiers = get_os_ids()
    package_map = load_yaml_file("/usr/lib/tuxedo-tomte-light/package_mapper.yaml")
    logger.debug("package map:")
    logger.debug(package_map)
    packages_final: set[str] = set()

    for package in packages:
        mapped_value = package  # default: package stays unchanged
        if package in package_map:
            # iterate through mapping entries of this package
            for os_key, replacement in package_map[package].items():
                if os_key in os_identifiers:
                    mapped_value = replacement
                    break
        if not is_already_installed(mapped_value):
            packages_final.add(mapped_value)
    logger.debug("final packages:")
    logger.debug(packages_final)
    return packages_final
